/**
 * i-net software provides programming examples for illustration only, without warranty
 * either expressed or implied, including, but not limited to, the implied warranties
 * of merchantability and/or fitness for a particular purpose. This programming example
 * assumes that you are familiar with the programming language being demonstrated and
 * the tools used to create and debug procedures. i-net software support professionals
 * can help explain the functionality of a particular procedure, but they will not modify
 * these examples to provide added functionality or construct procedures to meet your
 * specific needs.
 *
 * Copyright © 1999-2025 i-net software GmbH, Berlin, Germany.
**/
package export;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.nio.charset.Charset;
import java.util.Properties;

import com.inet.report.CacheRenderData;
import com.inet.report.EngineRenderData;
import com.inet.viewer.RenderData;

/**
 * This sample shows how you can export a report to the HTML file format with i-net Clear Reports and without using the
 * Java report viewer API. HTML export create multiple files (one for each report page and image) which must be saved
 * separately. If you like to export to one of the other file formats like PDF, RTF, ... then refer to the sample file
 * "samples/export/ExportWithoutViewer.java".
 */
public class ExportWithoutViewerToHTML {

    private RenderData           data;

    private byte[]               chunk;

    private int                  offset;

    private String               fileName        = null;

    private byte[]               fileData        = null;

    private String               directoryPrefix = null;

    private static final Charset UTF8            = Charset.forName( "UTF8" );

    /**
     * We create an HTML export object with the properties which rendered our report.<br>
     * @param props The engine report properties.
     * @param useCache true, if a cache should be used. Helpful for very large reports to prevent OutOfMemeoryError
     */
    public ExportWithoutViewerToHTML( Properties props, boolean useCache ) {
        props.put( "export_fmt", "htm" );
        this.data = useCache ? new CacheRenderData( props, null, null ) : new EngineRenderData( "" );
        data.getExportChunkCount( props );
    }

    /**
     * Here we check for the next file to read. If we get one we save its name and data and set the state to RECORD
     * otherwise we set the state to EOF.
     */
    private void readData() {
        int nameLength = readInt();
        if( nameLength != -1 ) {
            fileName = new String( chunk, offset, nameLength, UTF8 );
            offset += nameLength;
            int dataLength = readInt();
            fileData = new byte[dataLength];
            System.arraycopy( chunk, offset, fileData, 0, dataLength );
            offset += dataLength;
            if( directoryPrefix == null ) {
                directoryPrefix = fileName.substring( 0, fileName.lastIndexOf( '.' ) ) + "/";
            } else {
                fileName = directoryPrefix + fileName;
            }
        } else {
            fileName = null;
            fileData = null;
        }
    }

    /**
     * Reads a 4-byte number from the byte array
     * @return int read from byte array
     */
    private int readInt() {
        return (chunk[offset++] & 0xff) | (chunk[offset++] & 0xff) << 8 | (chunk[offset++] & 0xff) << 16 | chunk[offset++] << 24;
    }

    /**
     * Save all HTML files
     * @param directory the directory to save
     * @throws IOException if an error occurred during reading from input stream
     */
    public void save( File directory ) throws IOException {
        chunk = data.getNextExportChunk();
        while( chunk != null ) {

            do {
                readData();
                if( fileName == null ) {
                    break;
                }
                File file = new File( directory, fileName );
                file.getAbsoluteFile().getParentFile().mkdirs();
                FileOutputStream out = new FileOutputStream( file );
                out.write( fileData );
                out.close();
            } while( offset + 4 < chunk.length );

            chunk = data.getNextExportChunk();
            offset = 0;
        }
    }

    /**
     * Runs export
     * @param args first: report file, second: output dir, third: layout type
     * @throws IOException if an error occurred during reading from input stream
     */
    public static void main( String[] args ) throws IOException {
        Properties props = new Properties();
        String reportFile;
        if( args.length > 0 ) {
            reportFile = args[0];
        } else {
            reportFile = ExportWithoutViewerToHTML.class.getResource( "/samples/sample.rpt" ).toString();
        }
        props.setProperty( "report", reportFile );
        props.setProperty( "file", reportFile.substring( 0, reportFile.lastIndexOf( '.' ) ) + ".html" );

        String outputDir = args.length > 1 ? args[1] : ".";
        if( args.length > 2 ) {
            props.setProperty( "layout", args[2] );
        }
        ExportWithoutViewerToHTML export = new ExportWithoutViewerToHTML( props, false );
        export.save( new File( outputDir ) );
    }
}
